<?php
declare(strict_types=1);

require_once __DIR__ . '/auth.php';

/**
 * Sidebar menu map (GuardPro style)
 * - Each item supports: label, icon, url, roles, children
 * - roles: list of role_keys allowed (e.g. ['admin','hr'])
 */
function sidebar_menu_map(): array
{
    return [
        // ===== COMMON / HOME =====
        [
            'label' => 'Dashboard',
            'icon'  => 'fas fa-tachometer-alt',
            'url'   => role_home_url(),     // dynamic based on role
            'roles' => ['admin','hr','manager','supervisor','controller','guard'],
        ],

        // ===== ADMIN =====
        [
            'label' => 'Administration',
            'icon'  => 'fas fa-user-shield',
            'roles' => ['admin'],
            'children' => [
                ['label'=>'Users', 'icon'=>'fas fa-users', 'url'=>'/users/index.php', 'roles'=>['admin']],
                ['label'=>'Roles & Access', 'icon'=>'fas fa-key', 'url'=>'/admin/rbac.php', 'roles'=>['admin']], // future
                ['label'=>'Audit Logs', 'icon'=>'fas fa-clipboard-list', 'url'=>'/admin/audit.php', 'roles'=>['admin']], // future
                ['label'=>'System Settings', 'icon'=>'fas fa-cog', 'url'=>'/admin/settings.php', 'roles'=>['admin']], // future
            ],
        ],

        // ===== HR =====
        [
            'label' => 'HR & Guards',
            'icon'  => 'fas fa-id-badge',
            'roles' => ['admin','hr'],
            'children' => [
                ['label'=>'Employees', 'icon'=>'fas fa-user-tie', 'url'=>'/hr/employees/index.php', 'roles'=>['admin','hr']],
                ['label'=>'Register Guard', 'icon'=>'fas fa-user-plus', 'url'=>'/hr/register-guard.php', 'roles'=>['admin','hr']],
                ['label'=>'Leaves', 'icon'=>'fas fa-plane-departure', 'url'=>'/leaves/index.php', 'roles'=>['admin','hr']],
                ['label'=>'Leave Calendar', 'icon'=>'fas fa-calendar', 'url'=>'/leaves/calendar.php', 'roles'=>['admin','hr']],
            ],
        ],

        // ===== CLIENTS / SITES =====
        [
            'label' => 'Clients & Sites',
            'icon'  => 'fas fa-building',
            'roles' => ['admin','manager','supervisor','controller'],
            'children' => [
                ['label'=>'Clients', 'icon'=>'fas fa-handshake', 'url'=>'/clients/index.php', 'roles'=>['admin','manager']],
                ['label'=>'Sites', 'icon'=>'fas fa-map-marked-alt', 'url'=>'/clients/sites.php', 'roles'=>['admin','manager','supervisor','controller']], // you can create this
                ['label'=>'Assignments', 'icon'=>'fas fa-user-tag', 'url'=>'/clients/assignments.php', 'roles'=>['admin','manager','controller']], // future
            ],
        ],

        // ===== ROSTER =====
        [
            'label' => 'Roster',
            'icon'  => 'fas fa-calendar-alt',
            'roles' => ['admin','manager','supervisor','controller'],
            'children' => [
                ['label'=>'Weekly Roster', 'icon'=>'fas fa-th', 'url'=>'/roster/index.php', 'roles'=>['admin','manager','supervisor','controller']],
                ['label'=>'Generate / Assign', 'icon'=>'fas fa-magic', 'url'=>'/roster/generate.php', 'roles'=>['admin','manager','controller']],
                ['label'=>'Export', 'icon'=>'fas fa-file-export', 'url'=>'/roster/export.php', 'roles'=>['admin','manager','controller']],
            ],
        ],

        // ===== ATTENDANCE =====
        [
            'label' => 'Attendance',
            'icon'  => 'fas fa-user-clock',
            'roles' => ['admin','supervisor','manager','controller','hr'],
            'children' => [
                ['label'=>'Monitor', 'icon'=>'fas fa-eye', 'url'=>'/attendance/index.php', 'roles'=>['admin','supervisor','manager','controller','hr']],
                ['label'=>'Reports', 'icon'=>'fas fa-chart-line', 'url'=>'/attendance/reports.php', 'roles'=>['admin','manager','hr']], // future
            ],
        ],

        // ===== INCIDENTS =====
        [
            'label' => 'Incidents',
            'icon'  => 'fas fa-exclamation-triangle',
            'roles' => ['admin','manager','supervisor','controller','guard'],
            'children' => [
                ['label'=>'All Incidents', 'icon'=>'fas fa-list', 'url'=>'/incidents/index.php', 'roles'=>['admin','manager','supervisor','controller']],
                ['label'=>'My Reports', 'icon'=>'fas fa-file-alt', 'url'=>'/guard/my-incidents.php', 'roles'=>['guard']], // optional
                ['label'=>'Report Incident', 'icon'=>'fas fa-plus-circle', 'url'=>'/guard/report-incident.php', 'roles'=>['guard','supervisor']],
            ],
        ],

        // ===== PATROL =====
        [
            'label' => 'Patrol',
            'icon'  => 'fas fa-route',
            'roles' => ['admin','supervisor','manager','guard'],
            'children' => [
                ['label'=>'Routes', 'icon'=>'fas fa-map', 'url'=>'/patrol-routes/index.php', 'roles'=>['admin','manager']],
                ['label'=>'Checkpoints', 'icon'=>'fas fa-qrcode', 'url'=>'/patrol-routes/checkpoints.php', 'roles'=>['admin','manager']], // you may pass route_id
                ['label'=>'Monitor Patrols', 'icon'=>'fas fa-satellite-dish', 'url'=>'/supervisor/monitor-patrols.php', 'roles'=>['admin','supervisor']],
                ['label'=>'My Patrol', 'icon'=>'fas fa-walking', 'url'=>'/guard/patrol-dashboard.php', 'roles'=>['guard']],
            ],
        ],

        // ===== PAYROLL (basic hooks) =====
        [
            'label' => 'Payroll',
            'icon'  => 'fas fa-file-invoice-dollar',
            'roles' => ['admin','hr','manager'],
            'children' => [
                ['label'=>'Payroll Dashboard', 'icon'=>'fas fa-coins', 'url'=>'/payroll/index.php', 'roles'=>['admin','hr','manager']],
                ['label'=>'Overtime & Claims', 'icon'=>'fas fa-receipt', 'url'=>'/payroll/claims.php', 'roles'=>['admin','hr']], // future
            ],
        ],

        // ===== GUARD SELF-SERVICE =====
        [
            'label' => 'My Workspace',
            'icon'  => 'fas fa-user',
            'roles' => ['guard'],
            'children' => [
                ['label'=>'My Shifts', 'icon'=>'fas fa-calendar', 'url'=>'/guard/my-shifts.php', 'roles'=>['guard']],
                ['label'=>'My Attendance', 'icon'=>'fas fa-user-clock', 'url'=>'/guard/my-attendance.php', 'roles'=>['guard']],
                ['label'=>'Apply Leave', 'icon'=>'fas fa-plane', 'url'=>'/guard/apply-leave.php', 'roles'=>['guard']],
                ['label'=>'My Leaves', 'icon'=>'fas fa-clipboard-check', 'url'=>'/guard/my-leaves.php', 'roles'=>['guard']],
            ],
        ],
    ];
}

/**
 * Returns the home dashboard URL for the current user's role (priority-based)
 */
function role_home_url(): string
{
    $roles = current_user()['roles'] ?? [];
    $priority = ['admin','hr','manager','supervisor','controller','guard'];

    $role = 'guard';
    foreach ($priority as $rk) {
        if (in_array($rk, $roles, true)) { $role = $rk; break; }
    }

    return match ($role) {
        'admin'      => '/admin/dashboard.php',
        'hr'         => '/hr/dashboard.php',
        'manager'    => '/manager/dashboard.php',
        'supervisor' => '/supervisor/dashboard.php',
        'controller' => '/controller/dashboard.php',
        default      => '/guard/dashboard.php',
    };
}

/**
 * Filter menu items by user roles (recursive)
 */
function filter_menu_by_role(array $menu, array $roles): array
{
    $out = [];
    foreach ($menu as $item) {
        $allowed = array_intersect($roles, $item['roles'] ?? []);
        if (empty($allowed)) continue;

        if (!empty($item['children'])) {
            $item['children'] = filter_menu_by_role($item['children'], $roles);
            if (empty($item['children'])) {
                // If parent has no visible children, keep parent only if it has a URL
                if (empty($item['url'])) continue;
            }
        }

        $out[] = $item;
    }
    return $out;
}

/**
 * Active state helpers
 */
function is_active_url(string $url): bool
{
    $uri = $_SERVER['REQUEST_URI'] ?? '';
    $path = parse_url($uri, PHP_URL_PATH) ?: $uri;
    return $path === (BASE_URL . $url) || $path === $url || str_contains($path, $url);
}

function has_active_child(array $children): bool
{
    foreach ($children as $c) {
        if (!empty($c['url']) && is_active_url($c['url'])) return true;
        if (!empty($c['children']) && has_active_child($c['children'])) return true;
    }
    return false;
}