<?php
session_start();
header('Content-Type: application/json');

require_once __DIR__ . '/vendor/autoload.php';

use App\Core\Database;

$db = Database::connect();

try {

if (
empty($_POST['csrf_token']) ||
$_POST['csrf_token'] !== $_SESSION['csrf_token']
) {
throw new Exception('Invalid security token.');
}

$db = Database::connect();
$db->beginTransaction();

$ref = $_POST['booking_reference'];

$stmt = $db->prepare("
SELECT b.*, pr.pricing_type, pr.base_price,
       pr.hourly_rate, pr.minimum_hours
FROM bookings b
JOIN price_rules pr ON pr.service_id = b.service_id
WHERE b.booking_reference = ?
LIMIT 1
");
$stmt->execute([$ref]);
$booking = $stmt->fetch();

if (!$booking) {
throw new Exception('Booking not found.');
}

if ($booking['status'] !== 'pending') {
throw new Exception('Booking can no longer be modified.');
}

// ======================================
// UPDATED VALUES
// ======================================
$pickup_address  = $_POST['pickup_address'];
$dropoff_address = $_POST['dropoff_address'];
$pickup_datetime = $_POST['pickup_datetime'];
$flight_number   = $_POST['flight_number'] ?? null;
$zone_id         = $_POST['zone_id'] ?? $booking['zone_id'];
$hours           = $_POST['hours'] ?? $booking['hours'];

$pricing_type  = $booking['pricing_type'];
$base_price    = (float)$booking['base_price'];
$hourly_rate   = (float)$booking['hourly_rate'];
$minimum_hours = (int)$booking['minimum_hours'];

$old_price = $booking['final_price'];

$final_price = 0;

// ======================================
// RECALCULATE PRICE
// ======================================
if ($pricing_type === 'fixed') {

$final_price = $base_price;

}

elseif ($pricing_type === 'zone') {

$stmt = $db->prepare("SELECT price_modifier FROM zones WHERE id = ?");
$stmt->execute([$zone_id]);
$zone = $stmt->fetch();

if (!$zone) {
throw new Exception('Invalid zone selected.');
}

$final_price = (float)$zone['price_modifier'];

}

elseif ($pricing_type === 'hourly') {

if ($hours < $minimum_hours) {
throw new Exception("Minimum {$minimum_hours} hours required.");
}

$final_price = $hourly_rate * $hours;

}

if ($final_price <= 0) {
throw new Exception('Unable to recalculate price.');
}

// ======================================
// UPDATE BOOKING
// ======================================
$stmt = $db->prepare("
UPDATE bookings
SET pickup_address = ?,
dropoff_address = ?,
pickup_datetime = ?,
flight_number = ?,
zone_id = ?,
hours = ?,
estimated_price = ?,
final_price = ?
WHERE id = ?
");

$stmt->execute([
$pickup_address,
$dropoff_address,
$pickup_datetime,
$flight_number,
$zone_id ?: null,
$hours ?: null,
$final_price,
$final_price,
$booking['id']
]);

// ======================================
// TIMELINE ENTRY
// ======================================
$stmt = $db->prepare("
INSERT INTO booking_timeline
(booking_id, status, updated_by)
VALUES (?, ?, 'customer')
");

$status_message = "updated (price {$old_price} → {$final_price})";

$stmt->execute([
$booking['id'],
$status_message
]);

$db->commit();

echo json_encode([
'status' => 'success',
'ref' => $ref
]);

} catch (Exception $e) {

if (isset($db) && $db->inTransaction()) {
$db->rollBack();
}

echo json_encode([
'status' => 'error',
'message' => $e->getMessage()
]);
}
